#!/usr/bin/perl -w

use strict;

use Getopt::Long;

use IO::Socket;
use IO::Handle;
use IO::Select;
use Socket;
use Fcntl qw(:DEFAULT F_SETSIG);
use POSIX qw(:sys_wait_h setsid);

use NetSNMP::default_store (':all');
use NetSNMP::agent::default_store (':all');
use NetSNMP::agent (':all');
use NetSNMP::OID (':all');
use NetSNMP::ASN (':all');

my $opt_subagent = 1;
my $opt_readtimeout = 1;
my $opt_host = "127.0.0.1";
my $opt_port_default = "/tmp/agent-name.stats";
my $opt_port;
my $opt_interactive = 0;
my $opt_pidfile;
my $opt_help;

use Sys::Syslog; 
# perl 5.008004 does not define these macros
use constant LOG_ERR => "LOG_ERR";	
use constant LOG_CRIT => "LOG_CRIT";	
use constant LOG_INFO => "LOG_INFO";	

sub usage
{
	print STDERR "USAGE: $0 agent-name root-oid [options]\n"
	." --subagent     - Run as an AgentX subagent. default: $opt_subagent\n"
        ." --host         - Server host. default $opt_host\n"
        ." --port         - Server port. default $opt_port_default\n"
	." --readtimeout  - Client read timeout. default: $opt_readtimeout\n"
	." --interactive  - Do not fork from terminal. default: $opt_interactive\n"
	." --pidfile      - Pid file name.\n"
	." --help         - Show this help message\n"
	;
	exit 0;
}

my $result = GetOptions (
	"subagent=i"    => \$opt_subagent,
	"host=s"        => \$opt_host,
        "port=s"        => \$opt_port,
	"readtimeout=i" => \$opt_readtimeout,
	"interactive"   => \$opt_interactive,
	"pidfile=s"	=> \$opt_pidfile,
	"help"          => \$opt_help);

my $opt_name = $ARGV[0];
my $opt_rootoid = $ARGV[1];

if (not defined $opt_name)
{
	print STDERR "Missing agent name\n";
	usage();
}

if (not defined $opt_rootoid)
{
	print STDERR "Missing root oid\n";
	usage();
}

if (defined $opt_help)
{
	usage();
}

$opt_port = "/tmp/$opt_name.stats"
	unless defined $opt_port;

my %typemap = (
	"other" 	=> ASN_APPLICATION,
	"integer"	=> ASN_INTEGER,
	"string" 	=> ASN_OCTET_STR,
	"counter"	=> ASN_COUNTER,
	"gauge"		=> ASN_GAUGE,
	"timeticks"	=> ASN_TIMETICKS,
	"counter64"	=> ASN_COUNTER64
);

#my %errmap = (
#	2 => SNMP_ERR_NOSUCHNAME,
#	5 => SNMP_ERR_GENERR
#);

my $full_name = "$opt_name"."_snmp_agent";
my $cmdid = time;

my $client;
my $sel;
my $serveraddr;

killfromfile("TERM", $opt_pidfile) if defined ($opt_pidfile);

init_client();
	
if (not $opt_interactive)
{
	daemonize($full_name); 
}

updatepidfile($opt_pidfile) if defined ($opt_pidfile);

$SIG{TERM} = \&sigterm;
my $exit = 0;

run_with_watchdog();
#run_without_watchdog();

Log (LOG_INFO, "Done");

###############################################################################

sub init_client
{
	if (defined (atoi ($opt_port)))
	{
   		$client = new IO::Socket::INET(Proto => 'udp', Type => SOCK_DGRAM, Blocking => 0) or
        		logdie ("Create client socket failed: $!");
   
   		$client->bind(0, INADDR_ANY) or
        		logdie ("Bind client socket failed: $!");
   
   		$serveraddr = sockaddr_in($opt_port, inet_aton($opt_host));
	}
	else   # using unix sockets
	{
		$client = new IO::Socket::UNIX(Proto => AF_UNIX, Type => SOCK_DGRAM, Blocking => 0) or
			logdie ("Create client socket failed: $!");

		my $client_sock = "/tmp/.$full_name"."_$$.sock";
		unlink $client_sock;
		my $clientaddr = sockaddr_un($client_sock);
		$client->bind($clientaddr) or
			logdie ("Bind client socket failed: $!");

		$serveraddr = sockaddr_un($opt_port);
                chmod 666,$client_sock;
	}
   	
	$sel = new IO::Select $client;   
}

sub run_without_watchdog
{
   $SIG{PIPE} = sub {};
   run_agent();
}

sub run_with_watchdog
{
   socketpair(R, W, AF_UNIX, SOCK_STREAM, PF_UNSPEC);
   shutdown(R, 1); # no writing for reader
   shutdown(W, 0); # no reading for writer
   $SIG{IO} = \&sigio;

   use constant CHLDEXIT_MAGIC => 13;
   while (not $exit)
   {
   	my $pid = fork();
   	if ($pid > 0)
   	{
   		# parent
   		my $chldrc = watch_child($pid);
   		last if ($exit or $chldrc == CHLDEXIT_MAGIC);	
   	}
   	elsif ($pid == 0)
   	{
   		# child
   		init_child();
   		run_agent();
   		exit CHLDEXIT_MAGIC;
   	}
   	else
   	{
   		logdie ("Can't fork");
   	}
   }
}

sub watch_child
{
	my $pid = shift;
	while (not $exit)
	{
		if (waitpid($pid, WNOHANG) > 0)
		{
			# chiled died
			my $chldrc = $? >> 8;
			Log (LOG_ERR, "Watchdog: Child exit with rc $chldrc");
			return $chldrc;
		}
		sleep(1);
	}
	return -1;
}

sub init_child
{
	close R;
	my $p = $$;

	fcntl(W, F_SETFL, fcntl(W, F_GETFL, 0) | O_ASYNC) or
		logdie("init_child: F_SETFL falied");
	
	fcntl(W, F_SETSIG, 0) or
		logdie("init_child: F_SETDIG falied");

	fcntl(W, F_SETOWN, $p) or
		logdie("init_child: F_SETOWN falied");
}

sub run_agent
{
	# Currently, the NetSNMP perl module enables log to stderr, and 
	# doesn't expose a way to change this, so there is no logging to
	# sysLog from inside the snmp library. 
		  
	my $agent = new NetSNMP::agent(	'Name' => $opt_name, 
					'AgentX' => $opt_subagent) or
		logdie("run_agent: Failed to initialize NetSNMP::agent");
	
	my $rootoid = new NetSNMP::OID($opt_rootoid) or 
		logdie("run_agent: Failed to initialize NetSNMP::OID");

	my @rootoid = $rootoid->to_array();

	$agent->register($opt_name, $rootoid, \&my_snmp_handler) or
		logdie("run_agent: Failed to register $opt_name");

	while (not $exit)
	{
		$agent->agent_check_and_process(1);
	}

	$agent->shutdown();
}

sub my_snmp_handler
{
	my ($handler, $registration_info, $request_info, $requests) = @_;
	my $request;

	my $setoid;
	if ($request_info->getMode() == MODE_GET)
	{
		$setoid = undef;
	}
	elsif ($request_info->getMode() == MODE_GETNEXT)
	{
		$setoid = 1;
	}
	else
	{
#		for ($request = $requests; $request; $request = $request->next())
#		{
#			$request->setError($request_info, SNMP_ERR_NOTWRITABLE);
#		}
		return;
	}

	++$cmdid;
	my $opid = 0;
	my $cmd = "req: $cmdid\n";
	my $ops = '';

	for ($request = $requests; $request; $request = $request->next())
	{
		my $oid = $request->getOID();
		my @oid = $oid->to_array();
		my $oidstr = join('.',@oid);

		if ($request_info->getMode() == MODE_GET)
		{
			++$opid;
			$ops = $ops . "op: $opid,GET,$oidstr\n";
		}
		else # MODE_GETNEXT
		{
			++$opid;
			$ops = $ops . "op: $opid,GETNEXT,$oidstr\n";
		}
	}

	return unless $ops;

	my $req = $cmd . $ops;

	if (!$client->send ($req, 0, $serveraddr))
	{
		Log (LOG_ERR, "Send failed (port: $opt_port). $!");
#		for ($request = $requests; $request; $request = $request->next())
#		{
#			$request->setError($request_info, SNMP_ERR_GENERR);
#		}
		return;
	}

	my $found;
	my $resp;
	my @lines;

	while (!$found)
	{
		return unless $sel->can_read($opt_readtimeout);
		$client->recv ($resp, 65536);

		@lines = split '\n', $resp;

		next if ($#lines < 0);

		if ($lines [0] =~ /^err: (.+)/)
		{
			Log (LOG_ERR, "Response error: $lines[0].");
#			for ($request = $requests; $request; $request = $request->next())
#			{
#				$request->setError($request_info, SNMP_ERR_GENERR);
#			}
			return;
		}
		next unless ($lines [0] =~ /^resp: *([0-9]+)/);
		next unless ($cmdid == $1);
		
		shift @lines;
		$found = 1;
	}

	my ($id, $rc, $err, $oid, $type, $value, $ok);
	$opid = 0;
	for ($request = $requests; $request; $request = $request->next())
	{
		++$opid;
		$ok = 1;
		$err = '';

		if ($lines [0] =~ /^op: *(.*)$/)
		{
			($id,$rc,$err,$oid,$type,$value) = split ',', $1;

			if ($opid != $id)
			{
				$ok = undef;
				$rc = 5;
			}
			elsif ($rc != 0)
			{
				$ok = undef;
			}
			elsif (!exists $typemap{$type})
			{
				$ok = undef;
				$rc = 5;
			}
			else
			{
				$request->setValue($typemap{$type}, $value);
				$request->setOID($oid) if ($setoid);
			}
		}
		else
		{
			$rc = 5;
			$ok = undef;
			Log (LOG_ERR, "Op response error: $lines[0].");
		}

#		if (!$ok)
#		{
#			$request->setError($request_info, maperror($rc));
#		}
		shift @lines;
	}
}

sub atoi
{
	my $str = shift;
	my($num, $unparsed) = POSIX::strtod($str);
	return undef if ($unparsed);
	return int($num);
}

#sub maperror
#{
#	my $rc = shift;
#	return $errmap{$rc} if (exists $errmap{$rc});
#	return SNMP_ERR_GENERR;
#}

sub sigio
{
	Log (LOG_ERR, "got SIGIO");
	$exit = 1;
}

sub sigterm
{
	Log (LOG_ERR, "got SIGTERM");
	$exit = 1;
}

sub Log
{
	my ($level, $msg) = @_;	
	print STDERR "$msg\n";
	syslog ($level, "$msg") unless $opt_interactive;
}

sub logdie
{
	my $msg = shift;
	Log (LOG_CRIT, "$msg. $!");
	exit 1; # will cause child to respawn
}

sub daemonize
{
	my $name = shift;
	openlog($name, "ndelay,pid", "local0");

	#chdir '/'			or die "Can't chdir to /: $!";
	open STDIN, '/dev/null'		or die "Can't read /dev/null: $!";
	open STDOUT, '>>/dev/null'	or die "Can't write to /dev/null: $!";
	open STDERR, '>>/dev/null'	or die "Can't write to /dev/null: $!";
	defined(my $pid = fork)		or die "Can't fork: $!";
	exit if $pid;
	setsid				or die "Can't start a new session: $!";
	#umask 0;	
}

sub killfromfile
{
	my ($sig, $file) = @_;
	open(F, $file) or return;
	my $pid = <F>;
	chomp($pid);
	close F;
	my $cnt = kill ($sig, $pid);
	if ($cnt > 0 and $sig eq "TERM")
	{
		Log (LOG_INFO, "kill pervoius daemon. pid=$pid"); 
	}
}

sub updatepidfile
{
	my $file = shift;
	if (not open(F, ">$file"))
	{
		Log (LOG_ERR, "Can't update pid file $file: $!"); 
		return;
	}
	print F "$$\n";
	close F;
}

